import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { motion, AnimatePresence } from 'framer-motion'
import { Video, Calendar, CheckCircle2, ArrowRight, Monitor, Scissors, Play, Award, Users, Clock, Zap, Sparkles } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './Enroll.css'

const Enroll = () => {
  const navigate = useNavigate()
  const [selectedSoftware, setSelectedSoftware] = useState('')
  const [selectedDate, setSelectedDate] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)

  const softwareOptions = [
    { 
      id: 'premiere', 
      name: 'Premiere Pro', 
      icon: Monitor,
      color: '#6366F1',
      description: 'Профессиональный монтаж для кино и ТВ',
      features: ['Цветокоррекция', 'Аудио', 'Эффекты']
    },
    { 
      id: 'davinci', 
      name: 'DaVinci Resolve', 
      icon: Video,
      color: '#8B5CF6',
      description: 'Мощная система для цветокоррекции',
      features: ['Fusion', 'Fairlight', 'Color']
    },
    { 
      id: 'finalcut', 
      name: 'Final Cut', 
      icon: Scissors,
      color: '#EC4899',
      description: 'Быстрый монтаж для Mac',
      features: ['Magnetic Timeline', 'Multicam', 'HDR']
    },
    { 
      id: 'capcut', 
      name: 'CapCut', 
      icon: Play,
      color: '#10B981',
      description: 'Простой монтаж для соцсетей',
      features: ['Шаблоны', 'Эффекты', 'Музыка']
    }
  ]

  const today = new Date()
  const maxDate = new Date(today.getTime() + 180 * 24 * 60 * 60 * 1000)
  const minDateStr = today.toISOString().split('T')[0]
  const maxDateStr = maxDate.toISOString().split('T')[0]

  const handleSubmit = (e) => {
    e.preventDefault()
    if (!selectedSoftware || !selectedDate) return

    const saved = localStorage.getItem('videoProjects')
    const projects = saved ? JSON.parse(saved) : []
    
    const softwareData = softwareOptions.find(s => s.id === selectedSoftware)
    
    const newProject = {
      id: Date.now(),
      course: `${softwareData.name}: Базовый курс`,
      instructor: 'Преподаватель будет назначен',
      work: 'Первый проект',
      rating: null,
      photo: '/images/project-placeholder.jpg'
    }
    
    projects.push(newProject)
    localStorage.setItem('videoProjects', JSON.stringify(projects))
    
    setIsSubmitted(true)
    setTimeout(() => {
      navigate('/')
    }, 2000)
  }

  return (
    <div className="enroll-tech-page">
      <div className="container">
        <motion.section 
          className="enroll-header-tech"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ duration: 0.6 }}
        >
          <motion.div 
            className="enroll-badge-tech"
            initial={{ scale: 0 }}
            animate={{ scale: 1 }}
            transition={{ delay: 0.2, type: "spring" }}
          >
            <Video size={16} strokeWidth={2.5} />
            <span>НОВАЯ ЗАПИСЬ</span>
          </motion.div>
          <h1 className="enroll-title-tech">
            Записаться на <span className="title-gradient-tech">курс</span>
          </h1>
          <p className="enroll-description-tech">
            Выберите программное обеспечение и дату старта. Начните свой путь в видеомонтаже!
          </p>
        </motion.section>

        <div className="enroll-layout-tech">
          <motion.section 
            className="form-section-tech"
            initial={{ opacity: 0, x: -30 }}
            animate={{ opacity: 1, x: 0 }}
            transition={{ delay: 0.3 }}
          >
            <form className="form-tech" onSubmit={handleSubmit}>
              <div className="form-group-tech">
                <label className="form-label-tech">
                  <Monitor size={24} strokeWidth={2} />
                  <span>Выберите ПО</span>
                </label>
                <div className="software-list-tech">
                  {softwareOptions.map((software, index) => {
                    const Icon = software.icon
                    return (
                      <motion.button
                        key={software.id}
                        type="button"
                        className={`software-item-tech ${selectedSoftware === software.id ? 'selected' : ''}`}
                        onClick={() => setSelectedSoftware(software.id)}
                        initial={{ opacity: 0, x: -20 }}
                        animate={{ opacity: 1, x: 0 }}
                        transition={{ delay: 0.4 + index * 0.1 }}
                        whileHover={{ x: 5 }}
                        whileTap={{ scale: 0.98 }}
                        style={{ 
                          borderLeftColor: selectedSoftware === software.id ? software.color : 'var(--tech-border)',
                          background: selectedSoftware === software.id 
                            ? `linear-gradient(90deg, rgba(${software.color === '#6366F1' ? '99, 102, 241' : software.color === '#8B5CF6' ? '139, 92, 246' : software.color === '#EC4899' ? '236, 72, 153' : '16, 185, 129'}, 0.1), var(--tech-card))`
                            : 'var(--tech-card)'
                        }}
                      >
                        <div className="software-icon-wrapper-tech">
                          <div 
                            className="software-icon-tech"
                            style={{ 
                              background: selectedSoftware === software.id 
                                ? `linear-gradient(135deg, ${software.color}, ${software.color}dd)`
                                : 'var(--tech-surface)',
                              borderColor: software.color
                            }}
                          >
                            <Icon size={24} strokeWidth={2} />
                          </div>
                        </div>
                        <div className="software-content-tech">
                          <h3 className="software-name-tech">{software.name}</h3>
                          <p className="software-desc-tech">{software.description}</p>
                          <div className="software-features-tech">
                            {software.features.map((feature, idx) => (
                              <span key={idx} className="feature-tag-tech">{feature}</span>
                            ))}
                          </div>
                        </div>
                        {selectedSoftware === software.id && (
                          <motion.div 
                            className="check-mark-tech"
                            initial={{ scale: 0, rotate: -180 }}
                            animate={{ scale: 1, rotate: 0 }}
                            transition={{ type: "spring" }}
                          >
                            <CheckCircle2 size={24} strokeWidth={2.5} fill="currentColor" />
                          </motion.div>
                        )}
                      </motion.button>
                    )
                  })}
                </div>
              </div>

              <AnimatePresence>
                {selectedSoftware && (
                  <motion.div
                    className="form-group-tech date-group-tech"
                    initial={{ opacity: 0, height: 0 }}
                    animate={{ opacity: 1, height: 'auto' }}
                    exit={{ opacity: 0, height: 0 }}
                    transition={{ duration: 0.3 }}
                  >
                    <label className="form-label-tech">
                      <Calendar size={24} strokeWidth={2} />
                      <span>Дата старта</span>
                    </label>
                    <div className="date-input-wrapper-tech">
                      <Calendar size={20} strokeWidth={2} className="date-icon-tech" />
                      <input
                        type="date"
                        value={selectedDate}
                        onChange={(e) => setSelectedDate(e.target.value)}
                        min={minDateStr}
                        max={maxDateStr}
                        className="date-input-tech"
                        required
                      />
                    </div>
                  </motion.div>
                )}
              </AnimatePresence>

              <motion.button 
                type="submit" 
                className="submit-btn-tech"
                disabled={!selectedSoftware || !selectedDate}
                whileHover={{ scale: selectedSoftware && selectedDate ? 1.05 : 1 }}
                whileTap={{ scale: selectedSoftware && selectedDate ? 0.95 : 1 }}
              >
                <span>Записаться</span>
                <ArrowRight size={20} strokeWidth={2.5} />
              </motion.button>
            </form>

            {isSubmitted && (
              <motion.div 
                className="success-tech"
                initial={{ opacity: 0, scale: 0.8 }}
                animate={{ opacity: 1, scale: 1 }}
              >
                <CheckCircle2 size={36} strokeWidth={2} fill="currentColor" />
                <span>Запись оформлена! Переходим к вашим проектам...</span>
              </motion.div>
            )}
          </motion.section>

          <aside className="info-panel-tech">
            <motion.div 
              className="info-card-tech"
              initial={{ opacity: 0, x: 30 }}
              animate={{ opacity: 1, x: 0 }}
              transition={{ delay: 0.5 }}
            >
              <div className="info-header-tech">
                <Award size={28} strokeWidth={2} />
                <h3 className="info-title-tech">Преимущества</h3>
              </div>
              <ul className="benefits-list-tech">
                <li>
                  <Users size={20} strokeWidth={2} />
                  <span>Опытные преподаватели</span>
                </li>
                <li>
                  <Clock size={20} strokeWidth={2} />
                  <span>Гибкий график</span>
                </li>
                <li>
                  <Zap size={20} strokeWidth={2} />
                  <span>Практические проекты</span>
                </li>
                <li>
                  <Award size={20} strokeWidth={2} />
                  <span>Сертификат по окончании</span>
                </li>
              </ul>
            </motion.div>

            <motion.div 
              className="info-card-tech"
              initial={{ opacity: 0, x: 30 }}
              animate={{ opacity: 1, x: 0 }}
              transition={{ delay: 0.6 }}
            >
              <div className="info-header-tech">
                <Sparkles size={28} strokeWidth={2} />
                <h3 className="info-title-tech">Что вы получите</h3>
              </div>
              <div className="features-grid-tech">
                <div className="feature-box-tech">
                  <div className="feature-number-tech">01</div>
                  <p>Теоретические основы</p>
                </div>
                <div className="feature-box-tech">
                  <div className="feature-number-tech">02</div>
                  <p>Практические задания</p>
                </div>
                <div className="feature-box-tech">
                  <div className="feature-number-tech">03</div>
                  <p>Разбор ваших работ</p>
                </div>
                <div className="feature-box-tech">
                  <div className="feature-number-tech">04</div>
                  <p>Портфолио проектов</p>
                </div>
              </div>
            </motion.div>
          </aside>
        </div>

        <ImageSlider />

        <section className="testimonials-tech-section">
          <motion.h2 
            className="section-title-tech"
            initial={{ opacity: 0 }}
            whileInView={{ opacity: 1 }}
            viewport={{ once: true }}
          >
            <Video size={32} strokeWidth={2} />
            Отзывы студентов
          </motion.h2>
          <div className="testimonials-grid-tech">
            {[
              { text: 'Отличный курс по Premiere Pro! Научился профессиональному монтажу за короткое время.', author: 'Иван П.', rating: 5 },
              { text: 'DaVinci Resolve открыл для меня новые возможности в цветокоррекции. Рекомендую!', author: 'Елена К.', rating: 5 },
              { text: 'Преподаватели очень внимательные, всегда помогают с вопросами. Спасибо!', author: 'Александр М.', rating: 5 }
            ].map((testimonial, index) => (
              <motion.div 
                key={index}
                className="testimonial-card-tech"
                initial={{ opacity: 0, y: 30 }}
                whileInView={{ opacity: 1, y: 0 }}
                viewport={{ once: true }}
                transition={{ delay: index * 0.1 }}
                whileHover={{ y: -10 }}
              >
                <div className="testimonial-rating-tech">
                  {[...Array(testimonial.rating)].map((_, i) => (
                    <Sparkles key={i} size={16} strokeWidth={2} fill="currentColor" />
                  ))}
                </div>
                <p className="testimonial-text-tech">{testimonial.text}</p>
                <div className="testimonial-author-tech">— {testimonial.author}</div>
              </motion.div>
            ))}
          </div>
        </section>
      </div>
    </div>
  )
}

export default Enroll

